﻿using System;
using System.Collections;
using System.Collections.Generic;
using Server;
using Server.Items;
using Server.Engines.Lua;
using RemoteScripting;
using LuaInterface;
using Server.Gumps;
using Server.Mobiles;

namespace Server.Engines.XmlSpawner2
{
	class XmlLuaScript : XmlAttachment, ILuaCodeConsumer
	{
        private class FailsafeTimer : Timer
        {
            private Mobile m_Mob;
            private XmlLuaScript m_Script;

            public FailsafeTimer(XmlLuaScript script, Mobile mob, TimeSpan delay)
                : base(delay)
            {
                Priority = TimerPriority.OneSecond;
                m_Mob = mob;
                m_Script = script;
            }

            protected override void OnTick()
            {
                if (m_Script != null && m_Mob != null && m_Script.Lua != null)
                    LuaHelper.CallFunction(m_Script, "OnTimerExpired", new object[] { m_Mob });
            }
        }

        private LuaInterface.Lua m_Lua;
		private LuaCode m_Script;
		private string m_ScriptName;
        private Dictionary<Serial, FailsafeTimer> m_Timers = new Dictionary<Serial, FailsafeTimer>();

        #region ILuaCodeConsumer Members

        public LuaInterface.Lua Lua
		{
			get { return m_Lua; }
		}

		public LuaCode Script
		{
			get { return m_Script; }
			set
			{
				if ( m_Script != null )
				{
					m_Script.Unsubscribe( this );
				}

				m_Script = value;

				if ( m_Script != null )
				{
					m_Script.Subscribe( this );
				}
			}
		}

		[CommandProperty( AccessLevel.GameMaster )]
		public string ScriptName
		{
			get { return m_ScriptName; }
			set
			{
                if (value == null)
                {
                    Script = null;
                    m_ScriptName = null;
                }
                else if (LuaCodeBase.Scripts[value] != null)
                {
                    m_ScriptName = value;

                    Script = (LuaCode)LuaCodeBase.Scripts[m_ScriptName];
                }
                else
                    throw new Exception("Script name given does not exist in LuaCodeBase!");
			}
		}

		public string Description
		{
			get
			{
				if ( AttachedTo is Mobile )
				{
					Mobile mobile = AttachedTo as Mobile;

					return String.Format( "XmlLuaScript, Mobile '{0}', Type '{1}', Serial {2}", mobile.Name, mobile.GetType().ToString(), mobile.Serial.Value.ToString() );
				}
				else if ( AttachedTo is Item )
				{
					Item item = AttachedTo as Item;

					return String.Format( "XmlLuaScript, Item '{0}', Type '{1}', Serial {2}", item.Name, item.GetType().ToString(), item.Serial.Value.ToString() );
				}

				return "Some object";
			}
        }

        public void OnLostSubscription()
        {
            m_Lua = null;
            ScriptName = null;
            Update();
        }

        public void OnScriptChanged()
        {
            Update();
        }

        #endregion

        // this fixes a deserialization problem with Lua doing the update before XmlAttach has
        // fully deserialized the attachment. --cheetah2003
        public override object AttachedTo
        {
            get
            {
                return base.AttachedTo;
            }
            set
            {
                base.AttachedTo = value;
                Update();
            }
        }

		public void Update()
		{
            lock (m_Timers)
            {
                foreach (FailsafeTimer timer in m_Timers.Values)
                    timer.Stop();
            }
            m_Timers.Clear();

            if (m_Lua != null)
            {
                m_Lua.Dispose();
                m_Lua = null;
            }

            if (m_Script != null)
			{
				m_Lua = new LuaInterface.Lua();

				try
				{
					m_Lua["owner"] = this;
					m_Lua["this"] = AttachedTo;

					m_Lua.DoString( m_Script.Code );

					LuaHelper.CallFunction( this, "OnInitialize" );
				}
				catch ( Exception e )
				{
					LuaHelper.BroudcastError( this, e.Message );
				}
			}
		}

        public override void OnDelete()
        {
            Script = null;
            base.OnDelete();
        }

		public void OnError( string message )
		{

		}

		[Attachable]
		public XmlLuaScript()
		{
		}

		[CommandProperty( AccessLevel.GameMaster )]
		public override bool HandlesOnSpeech { get { return true; } }

		[CommandProperty( AccessLevel.GameMaster )]
		public override bool HandlesOnMovement { get { return true; } }

		[CommandProperty( AccessLevel.GameMaster )]
		public override bool HandlesOnKill { get { return true; } }

		[CommandProperty( AccessLevel.GameMaster )]
		public override bool HandlesOnKilled { get { return true; } }

		public override void OnAttach()
		{
			base.OnAttach();

			ArrayList attachments = XmlAttach.FindAttachments( AttachedTo, typeof( XmlLuaScript ) );

			if ( ( AttachedTo is Mobile || AttachedTo is Item ) && attachments.Count == 1 )
			{
                // this just crashes the server instantly when attaching. --cheetah2003
				// m_Lua["this"] = AttachedTo;

				LuaHelper.CallFunction( this, "OnAttach" );
			}
			else
			{
				Delete();
			}
		}

		public override bool BlockDefaultOnUse( Mobile from, object target )
		{
            return LuaHelper.CallFunction(this, "BlockDefaultOnUse", new object[] { from, target }).AsBoolean(0, false);
		}

		public override void OnUse( Mobile from )
		{
			LuaHelper.CallFunction( this, "OnUse", from );
		}

		public override void OnUser( object target )
		{
			LuaHelper.CallFunction( this, "OnUser", target );
		}

		public override void OnSpeech( SpeechEventArgs args )
		{
			LuaHelper.CallFunction( this, "OnSpeech", args );
		}

		public override void OnMovement( MovementEventArgs args )
		{
			LuaHelper.CallFunction( this, "OnMovement", args );
		}

		public override void OnKill( Mobile killed, Mobile killer )
		{
			LuaHelper.CallFunction( this, "OnKill", killed, killer );
		}

		public override void OnBeforeKill( Mobile killed, Mobile killer )
		{
			LuaHelper.CallFunction( this, "OnBeforeKill", killed, killer );
		}

		public override void OnKilled( Mobile killed, Mobile killer )
		{
			LuaHelper.CallFunction( this, "OnKilled", killed, killer );
		}

		public override void OnBeforeKilled( Mobile killed, Mobile killer )
		{
			LuaHelper.CallFunction( this, "OnBeforeKilled", killed, killer );
		}

		public override void OnWeaponHit( Mobile attacker, Mobile defender, BaseWeapon weapon, int damageGiven )
		{
			LuaHelper.CallFunction( this, "OnWeaponHit", attacker, defender, weapon, damageGiven );
		}

		public override int OnArmorHit( Mobile attacker, Mobile defender, Item armor, BaseWeapon weapon, int damageGiven )
		{
			return LuaHelper.CallFunction( this, "OnArmorHit", attacker, defender, armor, weapon, damageGiven ).AsInteger( 0, 0 );
		}

		public override string OnIdentify( Mobile from )
		{
			return LuaHelper.CallFunction( this, "OnIdentify", from ).AsString( 0, null );
		}

        public bool GenerateRewardItem(Type itemType, Container cont, int maxprops, int minintensity, int maxintensity)
        {
            LootPack loot = new LootPack(new LootPackEntry[] { new LootPackEntry(false, new LootPackItem[] { new LootPackItem(itemType, 100) }, 100.0, 1, 4, 0, 75) });
            loot.Generate(null, cont, false, 0);
            return true;
        }

        public bool GenerateRewardItem(String itemType, Container cont, int maxprops, int minintensity, int maxintensity)
        {
            Type t = ScriptCompiler.FindTypeByName(itemType, false);
            if (t != null)
                return GenerateRewardItem(t, cont, maxintensity, minintensity, maxintensity);
            else
                return false;
        }

        /// <summary>
        /// This generates a magic item of itemtype name, placing it in container cont.
        /// This is approximately the same intensity of a "average" magic item Lootpack.
        /// Use other functions of this name to directly specify intensities.
        /// </summary>
        /// <param name="itemType"></param>
        /// <param name="cont"></param>
        /// <returns></returns>
        public bool GenerateRewardItem(String itemType, Container cont)
        {
            return GenerateRewardItem(itemType, cont, 4, 0, 75);
        }

        /// <summary>
        /// Bring up an XmlSpawner type gump, see XmlDialog for documentation
        /// </summary>
        /// <param name="mob"></param>
        /// <param name="gumpstring"></param>
        public void ExecuteGump(Mobile mob, String gumpstring)
        {
            if (gumpstring == null || gumpstring.Length <= 0) return;

            string status_str = null;

            Server.Mobiles.XmlSpawner.SpawnObject TheSpawn = new Server.Mobiles.XmlSpawner.SpawnObject(null, 0);

            TheSpawn.TypeName = gumpstring;
            string substitutedtypeName = BaseXmlSpawner.ApplySubstitution(null, this, mob, gumpstring);
            string typeName = BaseXmlSpawner.ParseObjectType(substitutedtypeName);

            Point3D loc = new Point3D(0, 0, 0);
            Map map = null;

            if (AttachedTo is Mobile)
            {
                Mobile m = AttachedTo as Mobile;
                loc = m.Location;
                map = m.Map;
            }
            else
                if (AttachedTo is Item && ((Item)AttachedTo).Parent == null)
                {
                    Item i = AttachedTo as Item;
                    loc = i.Location;
                    map = i.Map;
                }

            if (typeName == "GUMP")
            {
                BaseXmlSpawner.SpawnTypeKeyword(this, TheSpawn, typeName, substitutedtypeName, true, mob, loc, map, new XmlGumpCallback(GumpResponse), out status_str);
                // hold processing until the gump response is completed

                //m_HoldProcessing = true;
            }
            else
            {
                status_str = "not a GUMP specification";
            }

            ReportError(mob, status_str);
        }

        public static void GumpResponse(Mobile from, object invoker, string response)
        {
            if (invoker is XmlLuaScript)
            {
                LuaHelper.CallFunction((XmlLuaScript)invoker, "OnGumpResponse", new object[] { from, response });
            }
        }

        public void ExecuteActions(Mobile mob, string actions)
        {
            string[] args = actions.Split(';');

            for (int i = 0; i < args.Length; ++i)
                _ExecuteAction(mob, args[i]);
        }

        public void ExecuteAction(Mobile mob, String action)
        {
            ExecuteActions(mob, action);
        }

        /// <summary>
        /// Execute an action via XmlSpawner, see XmlDialog for documentation.
        /// </summary>
        /// <param name="mob"></param>
        /// <param name="action"></param>
        private void _ExecuteAction(Mobile mob, string action)
        {
            if (action == null || action.Length <= 0) return;
            string status_str = null;
            Server.Mobiles.XmlSpawner.SpawnObject TheSpawn = new Server.Mobiles.XmlSpawner.SpawnObject(null, 0);

            TheSpawn.TypeName = action;
            string substitutedtypeName = BaseXmlSpawner.ApplySubstitution(null, this, mob, action);
            string typeName = BaseXmlSpawner.ParseObjectType(substitutedtypeName);

            Point3D loc = new Point3D(0, 0, 0);
            Map map = null;

            if (AttachedTo is Mobile)
            {
                Mobile m = AttachedTo as Mobile;
                loc = m.Location;
                map = m.Map;
            }
            else
                if (AttachedTo is Item && ((Item)AttachedTo).Parent == null)
                {
                    Item i = AttachedTo as Item;
                    loc = i.Location;
                    map = i.Map;
                }

            if (BaseXmlSpawner.IsTypeOrItemKeyword(typeName))
            {
                BaseXmlSpawner.SpawnTypeKeyword(AttachedTo, TheSpawn, typeName, substitutedtypeName, true, mob, loc, map, out status_str);
            }
            else
            {
                // its a regular type descriptor so find out what it is
                Type type = SpawnerType.GetType(typeName);
                try
                {
                    string[] arglist = BaseXmlSpawner.ParseString(substitutedtypeName, 3, "/");
                    object o = Server.Mobiles.XmlSpawner.CreateObject(type, arglist[0]);

                    if (o == null)
                    {
                        status_str = "invalid type specification: " + arglist[0];
                    }
                    else
                        if (o is Mobile)
                        {
                            Mobile m = (Mobile)o;
                            if (m is BaseCreature)
                            {
                                BaseCreature c = (BaseCreature)m;
                                c.Home = loc; // Spawners location is the home point
                            }

                            m.Location = loc;
                            m.Map = map;

                            BaseXmlSpawner.ApplyObjectStringProperties(null, substitutedtypeName, m, mob, AttachedTo, out status_str);
                        }
                        else
                            if (o is Item)
                            {
                                Item item = (Item)o;
                                BaseXmlSpawner.AddSpawnItem(null, AttachedTo, TheSpawn, item, loc, map, mob, false, substitutedtypeName, out status_str);
                            }
                }
                catch { }
            }
            ReportError(mob, status_str);
        }

        /// <summary>
        /// Do condition checks as per XmlSpawner and XmlDialog, see those for documentation.
        /// </summary>
        /// <param name="from"></param>
        /// <param name="condition"></param>
        /// <returns></returns>
        public bool CheckCondition(Mobile from, String condition)
        {
            string status_str;
            return BaseXmlSpawner.CheckPropertyString(null, this, condition, from, out status_str);
        }

        private void ReportError(Mobile mob, string status_str)
        {
            if (status_str != null && mob != null && !mob.Deleted && mob is PlayerMobile && mob.AccessLevel > AccessLevel.Player)
            {
                mob.SendMessage(33, String.Format("{0}:{1}", AttachedTo.GetType().Name, status_str));
            }
        }

        public void StartTimer(Mobile mob, int delay)
        {
            StopTimer(mob);
            m_Timers.Add(mob.Serial, new FailsafeTimer(this, mob, TimeSpan.FromSeconds(delay)));
            m_Timers[mob.Serial].Start();
        }

        public void StopTimer(Mobile mob)
        {
            if (m_Timers.ContainsKey(mob.Serial))
            {
                m_Timers[mob.Serial].Stop();
                m_Timers.Remove(mob.Serial);
            }
        }
		public XmlLuaScript( ASerial serial )
			: base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int)1 );

			writer.Write( (string)m_ScriptName );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 1:
					{
						ScriptName = reader.ReadString();

						break;
					}
			}
            Update();
		}
	}
}
