using System;
using System.Collections;
using Server;
using Server.Mobiles;
using Server.Multis;
using Server.Items;
using Server.ContextMenus;

namespace Server.Mobiles
{
	public class PetParrot : BaseCreature
	{				
		public override bool NoHouseRestrictions{ get{ return true; } }
		
		private DateTime m_Birth;
		
		[CommandProperty( AccessLevel.GameMaster)]
		public DateTime Birth
		{
			get { return m_Birth; }
			set { m_Birth = value; }
		} 
		
		[Constructable]
		public PetParrot() : this( DateTime.MinValue, null, 0 )
		{
		}
		
		[Constructable]
		public PetParrot( DateTime birth, string name, int hue ) : base( AIType.AI_Animal, FightMode.None, 10, 1, 0.2, 0.4 )
		{
			Title = "the parrot";			
			Body = 282;
			BaseSoundID = 0xBF;
			
			SetStr( 31, 47 );
			SetDex( 36, 60 );
			SetInt( 8, 20 );
			
			CantWalk = true;
			Blessed = true;
			
			if ( birth != DateTime.MinValue )
				m_Birth = birth;
			else
				m_Birth = DateTime.Now;
				
			if ( name != null )
				Name = name;
				
			if ( hue > 0 )
				Hue = hue;
		}
		
		public PetParrot( Serial serial ) : base( serial )
		{
		}				
		
		public override void GetProperties( ObjectPropertyList list )
		{
			base.GetProperties( list );
			
			int weeks = GetWeeks( m_Birth );
			
			if ( weeks == 1 )
				list.Add( 1072626 ); // 1 week old
			else if ( weeks > 1 )
				list.Add( 1072627, weeks.ToString() ); // ~1_AGE~ weeks old
		}
		
		public override bool CanBeRenamedBy( Mobile from )
		{
			if ( (int) from.AccessLevel > (int) AccessLevel.Player )
				return true;
		
			BaseHouse house = BaseHouse.FindHouseAt( this );

			if ( house != null && house.IsCoOwner( from ) )
				return true;
			else
				return false;
		}	
				
		public override void OnSpeech( SpeechEventArgs e )
		{
			base.OnSpeech( e );
			
			if ( Utility.RandomDouble() < 0.05 )
			{
				Say( e.Speech );
				PlaySound( 0xC0 );
			}
		}
		
		public override bool OnDragDrop( Mobile from, Item dropped )
		{
			if ( dropped is ParrotWafer )
			{
				dropped.Delete();
				
				switch ( Utility.Random( 9 ) )
				{
					case 0: Say( 1072602, "#" + Utility.RandomMinMax( 1012003, 1012010 ) ); break; // I just flew in from ~1_CITYNAME~ and boy are my wings tired!
					case 1: Say( 1072603 ); break; // Wind in the sails!  Wind in the sails!
					case 2: Say( 1072604 ); break; // Arrrr, matey!
					case 3: Say( 1072605 ); break; // Loot and plunder!  Loot and plunder!
					case 4: Say( 1072606 ); break; // I want a cracker!
					case 5: Say( 1072607 ); break; // I'm just a house pet!
					case 6: Say( 1072599 ); break; // Thank thee for the cracker!
					case 7: Say( 1072600 ); break; // Shiver me timbers!
					case 8: Say( 1072601 ); break; // Don't make me into a feathered hat!
				}
				
				PlaySound( Utility.RandomMinMax( 0xBF, 0xC3 ) );
				
				return true;
			}
			else
				return false;			
		}
		
		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version
			
			writer.Write( (DateTime) m_Birth );
		}
		
		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();
			
			m_Birth = reader.ReadDateTime();
		}
		
		public static int GetWeeks( DateTime birth )
		{
			TimeSpan span = DateTime.Now - birth;
			
			return (int) ( span.TotalDays / 7 );		
		}
	}	
}