using System;
using System.Collections;
using System.Collections.Generic;
using Server;
using Server.Gumps;
using Server.Network;
using Server.Targeting;

namespace Server.Items
{
	public abstract class BaseQuiver : BaseContainer, IDyable
	{
		public override bool DisplaysContent{ get{ return false; } }
		
		public virtual int InitMinHits { get { return 0; } }
		public virtual int InitMaxHits { get { return 0; } }
		public virtual int ArtifactRarity { get { return 0; } }
		public virtual Race RequiredRace { get { return null; } }

		private Mobile m_Crafter;
		private ArmorQuality m_Quality;

		private int m_Hits;
		private int m_MaxHits;
		private int m_LowerAmmoCost;

		private AosAttributes m_AosAttributes;
		private AosSkillBonuses m_AosSkillBonuses;

		private double m_Redux;
		private int m_Arrows;
		private int m_Bolts;
		private int m_MaxAmmo;      
		private int m_GlassSword;
		#region Public

		[CommandProperty(AccessLevel.GameMaster)]
		public int ReduxPercent
		{
			get { return (int)(m_Redux * 100); }
			set
			{
				if (value < 0)
					value = 0;
				if (value > 100)
					value = 100;
				m_Redux = ((double)value) / 100;
				DebugWeight();
				InvalidateProperties();
			}
		}

		[CommandProperty(AccessLevel.GameMaster)]
		public int Arrows { get { return m_Arrows; } set { m_Arrows = value; DebugWeight(); InvalidateProperties(); } }

		[CommandProperty(AccessLevel.GameMaster)]
		public int Bolts { get { return m_Bolts; } set { m_Bolts = value; DebugWeight(); InvalidateProperties(); } }

		[CommandProperty(AccessLevel.GameMaster)]
		public int MaxAmmo { get { return m_MaxAmmo; } set { m_MaxAmmo = value; InvalidateProperties(); } }
      
		[CommandProperty(AccessLevel.GameMaster)]
		public int GlassSword { get { return m_GlassSword; } set { m_GlassSword = value; DebugWeight(); InvalidateProperties(); } }

		[CommandProperty(AccessLevel.GameMaster)]
		public AosAttributes Attributes { get { return m_AosAttributes; } set { } }

		[CommandProperty(AccessLevel.GameMaster)]
		public AosSkillBonuses SkillBonuses { get { return m_AosSkillBonuses; } set { } }

		[CommandProperty(AccessLevel.GameMaster)]
		public Mobile Crafter { get { return m_Crafter; } set { m_Crafter = value; InvalidateProperties(); } }

		[CommandProperty(AccessLevel.GameMaster)]
		public int HitPoints
		{
			get { return m_Hits; }
			set
			{
				if (m_Hits == value)
					return;

				if (value > m_MaxHits)
					value = m_MaxHits;

				m_Hits = value;

				InvalidateProperties();
			}
		}

		[CommandProperty(AccessLevel.GameMaster)]
		public int MaxHitPoints
		{
			get { return m_MaxHits; }
			set { m_MaxHits = value; InvalidateProperties(); }
		}

		[CommandProperty(AccessLevel.GameMaster)]
		public ArmorQuality Quality
		{
			get { return m_Quality; }
			set { m_Quality = value; InvalidateProperties(); }
		}

		[CommandProperty(AccessLevel.GameMaster)]
		public int LowerAmmoCost { get { return m_LowerAmmoCost; } set { m_LowerAmmoCost = value; InvalidateProperties(); } }

		#endregion

		public BaseQuiver(int itemID)
			: base(itemID)
		{
			Weight = 0.0;
			Layer = Layer.MiddleTorso;
			m_AosAttributes = new AosAttributes(this);
			m_AosSkillBonuses = new AosSkillBonuses(this);
			m_Arrows = 0;
			m_Bolts = 0;
			m_GlassSword = 0;
			m_MaxAmmo = 10000;
		}
		
		private void DebugWeight()
		{
			double weight = ( (m_Arrows + m_Bolts) * 0.01 ) + m_GlassSword; // ( m_GlassSword * 5.0 );
			weight = weight - (weight * m_Redux);
			
			if ( weight < 0 )
				weight = 0;
			
			this.Weight = (int) weight;
			
			//             if (Parent is Item)
			//                (Parent as Item).UpdateTotals();
			//             else if (Parent is Mobile)
			//                (Parent as Mobile).UpdateTotals();
			//             else
			//                UpdateTotals();
		}
		
		public bool ConsumeAmmo( Type ammo, int amount )
		{
			if ( ammo == typeof( Arrow ) )
			{
				if ( m_Arrows >= amount )
				{
					m_Arrows -= amount;
					DebugWeight();
               InvalidateProperties();
					return true;
				}
				else
					return false;
			}
			
			if ( ammo == typeof( Bolt ) )
			{
				if ( m_Bolts >= amount )
				{
					m_Bolts -= amount;
					DebugWeight();
               InvalidateProperties();
					return true;
				}
				else
					return false;
			}
			
			return false;
		}

		#region Drop
		
		public bool DropAmmo( Mobile from, Item item )
		{
			if (from == null || item == null)
				return false;

			if (!(item is Arrow || item is Bolt || item is GlassSword))
				return false;
         
         if ( item is Arrow )
			{
				if ( m_Arrows + item.Amount > m_MaxAmmo )
					from.SendMessage( 1161, "Do toulce je mozne dat jeste maximalne {0} sipu.", m_MaxAmmo - m_Arrows );
				else
				{
					from.SendLocalizedMessage( 1072675, item.Amount.ToString()); //~1_QUANTITY~ pieces of ammo were transferred to the quiver due to low stock.
					m_Arrows += item.Amount;
					item.Delete();
					DebugWeight();
					from.SendSound( 0x48, GetWorldLocation() );
					
					if( from.HasGump( typeof(QuiverGump) ) )
						from.CloseGump( typeof(QuiverGump) );

					from.SendGump( new QuiverGump( this ) );
               
					return true;
				}
			}
			
			if ( item is Bolt )
			{
				if ( m_Bolts + item.Amount > m_MaxAmmo )
					from.SendMessage( 1161, "Do toulce je mozne dat jeste maximalne {0} sipek.", m_MaxAmmo - m_Bolts );
				else
				{
					from.SendLocalizedMessage(1072675, item.Amount.ToString()); //~1_QUANTITY~ pieces of ammo were transferred to the quiver due to low stock.
					m_Bolts += item.Amount;
					item.Delete();
					DebugWeight();
					from.SendSound( 0x48, GetWorldLocation() );
					
					if( from.HasGump( typeof(QuiverGump) ) )
						from.CloseGump( typeof(QuiverGump) );

					from.SendGump( new QuiverGump( this ) );
               
					return true;
				}
			}
         
			if ( item is GlassSword && item.Name == "glass sword" )
			{
				if ( m_GlassSword + 1 > m_MaxAmmo )
					from.SendMessage( 1161, "Do toulce je mozne dat maximalne {0} sklenenych mecu.", m_MaxAmmo );
				else
				{
					m_GlassSword++;
					item.Delete();
					DebugWeight();
					from.SendSound( 0x38E, GetWorldLocation() );
					
					if( from.HasGump( typeof(QuiverGump) ) )
						from.CloseGump( typeof(QuiverGump) );

					from.SendGump( new QuiverGump( this ) );

					return true;
				}
			}
         
			return false;
      }

		public override bool OnDragDrop(Mobile from, Item item)
		{
         DropAmmo( from, item );
			return false;
		}
		#endregion

		#region Equipping

		public override bool CanEquip(Mobile from)
		{
			if (RequiredRace != null && from.Race != RequiredRace)
			{
				if (RequiredRace == Race.Elf)
					from.SendLocalizedMessage(1072203); // Only Elves may use this.
				else
					from.SendMessage("Only {0} may use this.", RequiredRace.PluralName);

				return false;
			}

			return true;
		}

		// 		public override bool OnEquip( Mobile from )
		// 		{
		// 			if( Core.AOS )
		// 				m_AosSkillBonuses.AddTo( from );
		//
		// 			return base.OnEquip( from );
		// 		}
		//
		// 		public override void OnRemoved( object parent )
		// 		{
		// 			if( Core.AOS )
		// 				m_AosSkillBonuses.Remove();
		//
		// 			return;
		// 		}

		public override void OnAdded(object parent)
		{
			if (parent is Mobile)
			{
				Mobile from = (Mobile)parent;

				if (Core.AOS)
					m_AosSkillBonuses.AddTo(from);

				int strBonus = m_AosAttributes.BonusStr;
				int dexBonus = m_AosAttributes.BonusDex;
				int intBonus = m_AosAttributes.BonusInt;

				if (strBonus != 0 || dexBonus != 0 || intBonus != 0)
				{
					string modName = this.Serial.ToString();

					if (strBonus != 0)
						from.AddStatMod(new StatMod(StatType.Str, modName + "Str", strBonus, TimeSpan.Zero));

					if (dexBonus != 0)
						from.AddStatMod(new StatMod(StatType.Dex, modName + "Dex", dexBonus, TimeSpan.Zero));

					if (intBonus != 0)
						from.AddStatMod(new StatMod(StatType.Int, modName + "Int", intBonus, TimeSpan.Zero));
				}
				
				from.CheckStatTimers();
			}
			InvalidateProperties();

			base.OnAdded(parent);
		}

		public override void OnRemoved(object parent)
		{
			if (parent is Mobile)
			{
				Mobile from = (Mobile)parent;

				if (Core.AOS)
					m_AosSkillBonuses.Remove();

				string modName = this.Serial.ToString();

				from.RemoveStatMod(modName + "Str");
				from.RemoveStatMod(modName + "Dex");
				from.RemoveStatMod(modName + "Int");
			}
			
			base.OnRemoved(parent);

			InvalidateProperties();
		}
		#endregion

		#region Properties

		public override void GetProperties(ObjectPropertyList list)
		{
			base.GetProperties(list);
			
			//             list.Add(1075265, m_Arrows.ToString(), m_MaxAmmo.ToString()); //Ammo: ~1_QUANTITY~/~2_CAPACITY~ arrows
			//             list.Add(1075266, m_Bolts.ToString(), m_MaxAmmo.ToString()); //Ammo: ~1_QUANTITY~/~2_CAPACITY~ bolts
			list.Add( String.Format( "<basefont color=#9999FF>Arrows: {0} Bolts: {1}\nGlass Swords: {2}<basefont color=White>", m_Arrows, m_Bolts, m_GlassSword ));
			
			list.Add(1072789, Weight.ToString()); //Weight: ~1_WEIGHT~ stones

			if (m_Crafter != null)
				list.Add(1050043, m_Crafter.Name); // crafted by ~1_NAME~

			if (m_AosSkillBonuses != null)
				m_AosSkillBonuses.GetProperties(list);

			if (m_Quality == ArmorQuality.Exceptional)
				list.Add(1060636); // exceptional

			if (RequiredRace == Race.Elf)
				list.Add(1075086); // Elves Only

			if (ArtifactRarity > 0)
				list.Add(1061078, ArtifactRarity.ToString()); // artifact rarity ~1_val~
			
			int prop;

			if (ReduxPercent != 0)
				list.Add(1072210, ReduxPercent.ToString()); //Weight reduction: ~1_PERCENTAGE~%

			if ((prop = (m_AosAttributes.WeaponDamage)) != 0)
				list.Add(1060401, prop.ToString()); // damage increase ~1_val~%

			if ((prop = m_AosAttributes.DefendChance) != 0)
				list.Add(1060408, prop.ToString()); // defense chance increase ~1_val~%

			if ((prop = m_AosAttributes.EnhancePotions) != 0)
				list.Add(1060411, prop.ToString()); // enhance potions ~1_val~%

			if ((prop = m_AosAttributes.CastRecovery) != 0)
				list.Add(1060412, prop.ToString()); // faster cast recovery ~1_val~

			if ((prop = m_AosAttributes.CastSpeed) != 0)
				list.Add(1060413, prop.ToString()); // faster casting ~1_val~

			if ((prop = (m_AosAttributes.AttackChance)) != 0)
				list.Add(1060415, prop.ToString()); // hit chance increase ~1_val~%

			if ((prop = m_AosAttributes.BonusDex) != 0)
				list.Add(1060409, prop.ToString()); // dexterity bonus ~1_val~

			if ((prop = m_AosAttributes.BonusHits) != 0)
				list.Add(1060431, prop.ToString()); // hit point increase ~1_val~

			if ((prop = m_AosAttributes.BonusInt) != 0)
				list.Add(1060432, prop.ToString()); // intelligence bonus ~1_val~

			if ((prop = m_AosAttributes.LowerManaCost) != 0)
				list.Add(1060433, prop.ToString()); // lower mana cost ~1_val~%

			if ((prop = m_AosAttributes.LowerRegCost) != 0)
				list.Add(1060434, prop.ToString()); // lower reagent cost ~1_val~%

			if ((prop = (m_AosAttributes.Luck)) != 0)
				list.Add(1060436, prop.ToString()); // luck ~1_val~

			if ((prop = m_AosAttributes.BonusMana) != 0)
				list.Add(1060439, prop.ToString()); // mana increase ~1_val~

			if ((prop = m_AosAttributes.RegenMana) != 0)
				list.Add(1060440, prop.ToString()); // mana regeneration ~1_val~

			if ((prop = m_AosAttributes.NightSight) != 0)
				list.Add(1060441); // night sight

			if ((prop = m_AosAttributes.ReflectPhysical) != 0)
				list.Add(1060442, prop.ToString()); // reflect physical damage ~1_val~%

			if ((prop = m_AosAttributes.RegenStam) != 0)
				list.Add(1060443, prop.ToString()); // stamina regeneration ~1_val~

			if ((prop = m_AosAttributes.RegenHits) != 0)
				list.Add(1060444, prop.ToString()); // hit point regeneration ~1_val~

			if ((prop = m_AosAttributes.SpellChanneling) != 0)
				list.Add(1060482); // spell channeling

			if ((prop = m_AosAttributes.SpellDamage) != 0)
				list.Add(1060483, prop.ToString()); // spell damage increase ~1_val~%

			if ((prop = m_AosAttributes.BonusStam) != 0)
				list.Add(1060484, prop.ToString()); // stamina increase ~1_val~

			if ((prop = m_AosAttributes.BonusStr) != 0)
				list.Add(1060485, prop.ToString()); // strength bonus ~1_val~

			if ((prop = m_LowerAmmoCost) != 0)
				list.Add(1075208, prop.ToString()); // Lower Ammo Cost ~1_Percentage~%

			if ((prop = m_AosAttributes.WeaponSpeed) != 0)
				list.Add(1060486, prop.ToString()); // swing speed increase ~1_val~%

			/*
            int phys, fire, cold, pois, nrgy;

            GetDamageTypes( null, out phys, out fire, out cold, out pois, out nrgy );

            if( phys != 0 )
                list.Add( 1060403, phys.ToString() ); // physical damage ~1_val~%

            if( fire != 0 )
                list.Add( 1060405, fire.ToString() ); // fire damage ~1_val~%

            if( cold != 0 )
                list.Add( 1060404, cold.ToString() ); // cold damage ~1_val~%

            if( pois != 0 )
                list.Add( 1060406, pois.ToString() ); // poison damage ~1_val~%

            if( nrgy != 0 )
                list.Add( 1060407, nrgy.ToString() ); // energy damage ~1_val~%
			 */

			if (m_Hits >= 0 && m_MaxHits > 0)
				list.Add(1060639, "{0}\t{1}", m_Hits, m_MaxHits); // durability ~1_val~ / ~2_val~
				
	     // mod to display attachment properties
	     Server.Engines.XmlSpawner2.XmlAttach.AddAttachmentProperties(this, list);
		}
		
		public bool Dye( Mobile from, DyeTub sender )
		{
			if ( Deleted )
				return false;

			Hue = sender.DyedHue;

			return true;
		}

		public override void OnDoubleClick(Mobile from)
		{
			if( from.HasGump( typeof(QuiverGump) ) )
				from.CloseGump( typeof(QuiverGump) );

			from.SendGump( new QuiverGump( this ) );
			return;
		}
		public BaseQuiver(Serial serial) : base(serial) { }
		#endregion

		#region SerDeser

		public override void Serialize(GenericWriter writer)
		{
			base.Serialize(writer);
			writer.Write((int)2); // version

			writer.Write( m_GlassSword );
         
         writer.Write( m_Arrows );
			writer.Write( m_Bolts );
			writer.Write( m_MaxAmmo );
			writer.Write( (int)m_Quality );

			writer.Write((double)m_Redux);
			writer.Write((int)m_LowerAmmoCost);
			m_AosAttributes.Serialize(writer);
			m_AosSkillBonuses.Serialize(writer);
			writer.Write(m_Crafter);
		}

		public override void Deserialize(GenericReader reader)
		{
			base.Deserialize(reader);
			int version = reader.ReadInt();

			switch (version)
			{
				case 2:
					{
						m_GlassSword = reader.ReadInt();
						goto case 1;
					}
            case 1:
					{
						m_Arrows = reader.ReadInt();
						m_Bolts = reader.ReadInt();
						m_MaxAmmo = reader.ReadInt();
						m_Quality = (ArmorQuality)reader.ReadInt();
						goto case 0;
					}
				case 0:
					{
						m_Redux = reader.ReadDouble();
						m_LowerAmmoCost = reader.ReadInt();
						m_AosAttributes = new AosAttributes(this, reader);
						m_AosSkillBonuses = new AosSkillBonuses(this, reader);
						m_Crafter = reader.ReadMobile();
						break;
					}
			}
			
  			if ( m_AosSkillBonuses == null )
  				m_AosSkillBonuses = new AosSkillBonuses( this );
  				
  			if ( Core.AOS && Parent is Mobile )
  				m_AosSkillBonuses.AddTo( (Mobile)Parent );
  				
  			int strBonus = m_AosAttributes.BonusStr;
  			int dexBonus = m_AosAttributes.BonusDex;
  			int intBonus = m_AosAttributes.BonusInt;
  
  			if ( Parent is Mobile && (strBonus != 0 || dexBonus != 0 || intBonus != 0) )
  			{
  				Mobile m = (Mobile)Parent;
  
  				string modName = Serial.ToString();
  
  				if ( strBonus != 0 )
  					m.AddStatMod( new StatMod( StatType.Str, modName + "Str", strBonus, TimeSpan.Zero ) );
  
  				if ( dexBonus != 0 )
  					m.AddStatMod( new StatMod( StatType.Dex, modName + "Dex", dexBonus, TimeSpan.Zero ) );
  
  				if ( intBonus != 0 )
  					m.AddStatMod( new StatMod( StatType.Int, modName + "Int", intBonus, TimeSpan.Zero ) );
  			}
  
  			if ( Parent is Mobile )
  				((Mobile)Parent).CheckStatTimers();
			
// 			if ( version < 1 )
// 			{
// 				List<Item> items = this.Items;
// 				ArrayList toDelete = new ArrayList();;
// 				m_Arrows = 0;
// 				m_Bolts = 0;
// 				
// 				for ( int a = 0; a < items.Count; ++a )
// 				{
// 					Item obj = items[a];
// 					
// 					if ( obj != null && (obj is Arrow || obj is Bolt) )
// 					{
// 						if ( obj is Arrow )
// 							m_Arrows += obj.Amount;
// 						
// 						if ( obj is Bolt )
// 							m_Bolts += obj.Amount;
// 						
// 						toDelete.Add( obj );
// 					}
// 				}
// 				
// 				foreach ( Item item in toDelete )
// 					item.Delete();
// 				
// 				m_Quality = ArmorQuality.Regular;
// 				if ( this is ElvenQuiver )
// 				  m_MaxAmmo = 5000;
// 				else
// 				  m_MaxAmmo = 10000;
// 			}
		}
		#endregion
		
    	private class QuiverTarget : Target
    	{
    		private BaseQuiver m_Quiver;
    
    		public QuiverTarget( BaseQuiver quiver ) : base( 1, false, TargetFlags.None )
    		{
    			m_Quiver = quiver;
    		}
    
    		protected override void OnTarget( Mobile from, object target )
    		{
    			if ( m_Quiver != null && target is Item && m_Quiver.DropAmmo( from, (Item)target ) )
    				from.Target = new QuiverTarget( m_Quiver );
    			else
            {
    				from.SendMessage( "Do toulce lze vlozit pouze sipy, sipky a sklenene mece." );
               from.SendGump( new QuiverGump( m_Quiver ) );
            }
    		}
    	}
		
		public class QuiverGump : Gump
		{

			private BaseQuiver m_Quiver;
			
			public QuiverGump( BaseQuiver quiver ) : base( 70, 70 )
			{
				m_Quiver = quiver;
				
				if ( m_Quiver == null )
					return;
				
				
				this.Closable=true;
				this.Disposable=true;
				this.Dragable=true;
				this.Resizable=false;

				AddPage(0);
            AddBackground(0, 0, 190, 256, 9200);
				AddImage(5, 5, 65);
				AddItem(25, 51, 3904);
				AddTooltip( 1023904 ); //arrows
				AddItem(25, 85, 7164);
				AddTooltip( 1027164 ); //crossbow bolts
				AddTextEntry(70, 55, 61, 20, 1152, 1, m_Quiver.Arrows.ToString());
				AddTextEntry(70, 91, 61, 20, 1152, 2, m_Quiver.Bolts.ToString());
				AddButton(132, 57, 5601, 5605, 1, GumpButtonType.Reply, 0);
				AddTooltip( 1023904 ); //arrows
				AddButton(132, 92, 5601, 5605, 2, GumpButtonType.Reply, 0);
				AddTooltip( 1027164 ); //crossbow bolts
            
				AddItem(0, 185, 9935, 91);
				AddTooltip( 1022316 ); //glass sword
            AddLabel( 63, 198, 0, String.Format( "Pocet: {0}", m_Quiver.GlassSword ) );
				AddButton(132, 200, 5601, 5605, 4, GumpButtonType.Reply, 0);
				AddTooltip( 1022316 ); //glass sword
            
            AddLabel(65, 228, 0, @"Vlozit");
				AddButton(132, 230, 5601, 5605, 3, GumpButtonType.Reply, 0);
				AddTooltip( 3000175 ); //Add
			}

			public override void OnResponse(NetState sender, RelayInfo info)
			{
				Mobile from = sender.Mobile;
				
				if ( m_Quiver == null || m_Quiver.Deleted )
					return;
    		   else if ( m_Quiver.RootParent != from )
    			{
                from.SendMessage( "Musis mit {0} u sebe!", m_Quiver.Name );
                
    			   if( from.HasGump( typeof(QuiverGump) ) )
    				  from.CloseGump( typeof(QuiverGump) );

    				return;
             }
         
				switch( info.ButtonID )
				{
					case 0:
						{
							break;
						}
					case 1:
						{
							TextRelay tr_Arrorws = info.GetTextEntry(1);
							if (tr_Arrorws != null)
							{
								int i_Ammo = 0;
								try
								{
									i_Ammo = Convert.ToInt32(tr_Arrorws.Text, 10);
								}
								catch
								{
									from.SendMessage(1161, "Musis napsat pouze cisla");
								}
								if (i_Ammo > 0)
								{
									if ( i_Ammo <= m_Quiver.Arrows )
									{
										if (i_Ammo <= m_Quiver.MaxAmmo)
										{
											from.AddToBackpack( new Arrow(i_Ammo) );
											from.SendMessage( 1161, "Vyjmul{1} jsi z toulce {0} sipu.", i_Ammo, (from.Female) ? "a" : "" );
											m_Quiver.Arrows -= i_Ammo;
                                 from.PlaySound( 0x48 );
										}
										else
											from.SendMessage(1161, "{0} je maximalni mnozstvi streliva, ktere lze vyjmout najednou!", m_Quiver.MaxAmmo);
									}
									else
										from.SendMessage(1173, "Nemas v toulci tolik sipu!");
								}
								from.SendGump( new QuiverGump( m_Quiver ) );
							}
							break;
						}
					case 2:
						{
							TextRelay tr_Bolts = info.GetTextEntry(2);
							if (tr_Bolts != null)
							{
								int i_Ammo = 0;
								try
								{
									i_Ammo = Convert.ToInt32(tr_Bolts.Text, 10);
								}
								catch
								{
									from.SendMessage(1161, "Musis napsat pouze cisla");
								}
								if (i_Ammo > 0)
								{
									if ( i_Ammo <= m_Quiver.Bolts )
									{
										if (i_Ammo <= m_Quiver.MaxAmmo)
										{
											from.AddToBackpack( new Bolt(i_Ammo) );
											from.SendMessage( 1161, "Vyjmul{1} jsi z toulce {0} sipek.", i_Ammo, (from.Female) ? "a" : "" );
											m_Quiver.Bolts -= i_Ammo;
                                 from.PlaySound( 0x48 );
										}
										else
											from.SendMessage(1161, "{0} je maximalni mnozstvi streliva, ktere lze vyjmout najednou!", m_Quiver.MaxAmmo);
									}
									else
										from.SendMessage(1173, "Nemas v toulci tolik sipek!");
								}
								from.SendGump( new QuiverGump( m_Quiver ) );
							}
							break;
						}
					case 3:
						{
                     from.Target = new QuiverTarget( m_Quiver );
							break;
						}
					case 4:
						{
                     if ( m_Quiver.GlassSword > 0 )
                     {
                        from.AddToBackpack( new GlassSword() );
                        m_Quiver.GlassSword--;
                        from.PlaySound( 571 );
                     }
                     
                     from.SendGump( new QuiverGump( m_Quiver ) );
							break;
						}
				}
			}
		}
	}
}