using System;
using System.Collections;
using Server.Items;

namespace Server.Mobiles
{
	public class ShadowFiend : BaseCreature
	{
		public override bool DeleteCorpseOnDeath{ get{ return true; } }
		public override bool BardImmune { get { return true; } }
		public override bool Unprovokable { get { return true; } }
		public override bool Uncalmable { get { return true; } }
		public override Poison PoisonImmune{ get{ return Poison.Lethal; } }
		public override bool IsNotScaredOfDeath{ get{ return true; } } //neutika pri malem poctu hp
		
		public override WeaponAbility GetWeaponAbility()
		{
			return Utility.RandomBool() ? WeaponAbility.ParalyzingBlow : WeaponAbility.CrushingBlow;
		}
		
		private UnhideTimer m_Timer;

		[Constructable]
		public ShadowFiend() : base( AIType.AI_Melee, FightMode.Closest, 10, 1, 0.2, 0.4 )
		{
			Name = "a shadow fiend";
			Body = 0xA8;

			// this to allow shadow fiend to loot from corpses
			Backpack backpack = new Backpack();
			backpack.Movable = false;
			AddItem( backpack );

			SetStr( 516, 605 );
			SetDex( 106, 115 );
			SetInt( 266, 305 );

			SetHits( 600, 700 );
			SetMana( 2500 );

			SetDamage( 10, 25 );

			SetDamageType( ResistanceType.Cold, 70 );
			SetDamageType( ResistanceType.Physical, 15 );
			SetDamageType( ResistanceType.Energy, 15 );

			SetResistance( ResistanceType.Physical, 100 );
			SetResistance( ResistanceType.Fire, 50, 70 );
			SetResistance( ResistanceType.Cold, 20, 50 );
			SetResistance( ResistanceType.Poison, 100 );
			SetResistance( ResistanceType.Energy, 40, 70 );

			SetSkill( SkillName.EvalInt, 77.6, 110.5 );
			SetSkill( SkillName.Necromancy, 120.6, 130.5 );
			SetSkill( SkillName.SpiritSpeak, 120.1, 150.5 );
			SetSkill( SkillName.Magery, 90.1, 120.1);
			SetSkill( SkillName.Poisoning, 80.5 );
			SetSkill( SkillName.Meditation, 110.0 );
			SetSkill( SkillName.MagicResist, 80.1, 85.0 );
			SetSkill( SkillName.Parry, 90.1, 95.1 );
			SetSkill( SkillName.Tactics, 100.0 );
			SetSkill( SkillName.Wrestling, 90.1, 120.0 );
			SetSkill( SkillName.Anatomy, 100.0 );

			Fame = 1000;
			Karma = -1000;

			m_Timer = new UnhideTimer( this );
			m_Timer.Start();
		}

		public override int GetIdleSound()
		{
			return 0x37A;
		}

		public override int GetAngerSound()
		{
			return 0x379;
		}

		public override int GetDeathSound()
		{
			return 0x381;
		}

		public override int GetAttackSound()
		{
			return 0x37F;
		}

		public override int GetHurtSound()
		{
			return 0x380;
		}

		public override bool CanRummageCorpses{ get{ return false; } }

		public override bool OnBeforeDeath()
		{
			if ( Backpack != null )
			{
				Backpack.DropItem( new Gold( Utility.RandomMinMax( 200, 320 ) ) );
            Backpack.Destroy();
         }

			Effects.SendLocationEffect( Location, Map, 0x376A, 10, 1 );
			return true;
		}
		
		public override void AlterMeleeDamageFrom(Mobile from, ref int damage)
		{
			damage = 0;
		}
		
		public override void OnKilledBy( Mobile mob )
		{
			base.OnKilledBy( mob );
			
			if ( Paragon.CheckArtifactChance( mob, this ) && mob.Talisman is KhalAnkurAmulet )
			{
				((KhalAnkurAmulet) mob.Talisman).SpendingPoints++;
			}
		}

		public ShadowFiend( Serial serial ) : base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );
			writer.Write( (int) 0 );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );
			int version = reader.ReadInt();

			m_Timer = new UnhideTimer( this );
			m_Timer.Start();
		}

		public override void OnAfterDelete()
		{
			if ( m_Timer != null )
				m_Timer.Stop();

			m_Timer = null;

			base.OnAfterDelete();
		}

		private class UnhideTimer : Timer
		{
			private ShadowFiend m_Owner;

			public UnhideTimer( ShadowFiend owner ) : base( TimeSpan.FromSeconds( 30.0 ) )
			{
				m_Owner = owner;
				Priority = TimerPriority.OneSecond;
			}

			protected override void OnTick()
			{
				if ( m_Owner.Deleted )
				{
					Stop();
					return;
				}

				foreach ( Mobile m in m_Owner.GetMobilesInRange( 10 ) )
				{
					if ( m != m_Owner && m.Player && m_Owner.CanBeHarmful( m ) && m.AccessLevel == AccessLevel.Player && m_Owner.InLOS( m ) )
					{
						m.RevealingAction();
						m.FixedParticles( 0x373A, 1, 17, 0x7E2, EffectLayer.Waist );
						m.PlaySound( 505 );
						
						m.Mana -= ( m.ManaMax * 20 ) / 100;
						m.Stam -= ( m.StamMax * 20 ) / 100;
					}
				}
			}
		}
	}
}
