using System;
using Server;
using Server.Gumps;
using Server.Network;
using System.Collections;
using Server.ContextMenus;
using Server.Targeting;
using System.Collections.Generic;

namespace Server.Items
{
	public class MiniHouseAddon : BaseAddon
	{
		private MiniHouseType m_Type;

		[CommandProperty( AccessLevel.GameMaster )]
		public MiniHouseType Type
		{
			get{ return m_Type; }
			set{ m_Type = value; Construct(); }
		}

		public override BaseAddonDeed Deed{ get{ return new MiniHouseDeed( m_Type ); } }

		[Constructable]
		public MiniHouseAddon() : this( MiniHouseType.StoneAndPlaster )
		{
		}

		[Constructable]
		public MiniHouseAddon( MiniHouseType type )
		{
			m_Type = type;

			Construct();
		}

		public void Construct()
		{
			Components.Clear();

			MiniHouseInfo info = MiniHouseInfo.GetInfo( m_Type );

			int size = (int)Math.Sqrt( info.Graphics.Length );
			int num = 0;

			for ( int y = 0; y < size; ++y )
				for ( int x = 0; x < size; ++x )
				AddComponent( new AddonComponent( info.Graphics[num++] ), size - x - 1, size - y - 1, 0 );
		}

		public MiniHouseAddon( Serial serial ) : base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version

			writer.Write( (int) m_Type );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 0:
					{
						m_Type = (MiniHouseType)reader.ReadInt();
						break;
					}
			}
		}
	}

	public class MiniHouseDeed : BaseAddonDeed
	{
		private MiniHouseType m_Type;

		[CommandProperty( AccessLevel.GameMaster )]
		public MiniHouseType Type
		{
			get{ return m_Type; }
			set{ m_Type = value; InvalidateProperties(); }
		}

		public override BaseAddon Addon{ get{ return new MiniHouseAddon( m_Type ); } }
		public override int LabelNumber{ get{ return 1062096; } } // a mini house deed

		public override void GetProperties( ObjectPropertyList list )
		{
			base.GetProperties( list );

			if ( m_Type != MiniHouseType.NoSelected )
				list.Add( MiniHouseInfo.GetInfo( m_Type ).LabelNumber );
		}

		[Constructable]
		public MiniHouseDeed() : this( MiniHouseType.NoSelected )
		{
		}

		[Constructable]
		public MiniHouseDeed( MiniHouseType type )
		{
			m_Type = type;

			Weight = 1.0;
			LootType = LootType.Blessed;
		}

		public override void OnDoubleClick( Mobile from )
		{
			if ( IsChildOf( from.Backpack ) )
			{
				if ( m_Type != MiniHouseType.NoSelected )
					from.Target = new TargetInternal( this );
				else
					from.SendGump( new MiniHouseDeedGump( this ) );
			}
			else
				from.SendLocalizedMessage( 1042001 ); // That must be in your pack for you to use it.
		}
		
		private class MiniHouseDeedGump : Gump
		{
			private MiniHouseDeed m_MiniHouse;

			public MiniHouseDeedGump( MiniHouseDeed minihouse ) : base( 150, 200 )
			{
				m_MiniHouse = minihouse;

				this.Closable=true;
				this.Disposable=true;
				this.Dragable=true;
				this.Resizable=false;
				this.AddPage(0);
				this.AddBackground(0, 0, 390, 379, 9200);
				this.AddItem(32, 30, 8900);
				this.AddItem(32, 70, 8926);
				this.AddItem(32, 110, 8927);
				this.AddItem(32, 150, 8905);
				this.AddItem(32, 190, 8928);
				this.AddItem(32, 230, 8929);
				this.AddItem(340, 190, 8949);
				this.AddLabel(220, 330, 1152, @"Castle");
				this.AddLabel(220, 285, 1152, @"Small Stone Keep");
				this.AddItem(330, 30, 8955);
				this.AddItem(220, 190, 8960);
				this.AddItem(330, 110, 8947);
				this.AddItem(32, 270, 8950);
				this.AddItem(32, 320, 8948);
				this.AddRadio(10, 40, 209, 208, true, (int)Buttons.StoneAndPlaster);
				this.AddRadio(80, 120, 209, 208, false, (int)Buttons.TwoStoryWoodAndPlaster);
				this.AddRadio(80, 40, 209, 208, false, (int)Buttons.Brick);
				this.AddRadio(10, 331, 209, 208, false, (int)Buttons.SmallMarbleWorkshop);
				this.AddRadio(10, 284, 209, 208, false, (int)Buttons.SmallStoneWorkshop);
				this.AddRadio(10, 240, 209, 208, false, (int)Buttons.ThatchedRoof);
				this.AddRadio(10, 200, 209, 208, false, (int)Buttons.WoodAndPlaster);
				this.AddRadio(10, 160, 209, 208, false, (int)Buttons.Wooden);
				this.AddRadio(10, 120, 209, 208, false, (int)Buttons.SmallBrick);
				this.AddRadio(10, 80, 209, 208, false, (int)Buttons.FieldStone);
				this.AddRadio(190, 40, 209, 208, false, (int)Buttons.LargeHouseWithPatio);
				this.AddRadio(78, 284, 209, 208, false, (int)Buttons.Tower);
				this.AddRadio(80, 200, 209, 208, false, (int)Buttons.TwoStoryStoneAndPlaster);
				this.AddRadio(301, 40, 209, 208, false, (int)Buttons.TwoStoryLogCabin);
				this.AddRadio(190, 331, 209, 208, false, (int)Buttons.Castle);
				this.AddRadio(190, 284, 209, 208, false, (int)Buttons.SmallStoneKeep);
				this.AddRadio(190, 200, 209, 208, false, (int)Buttons.TwoStoryVilla);
				this.AddRadio(190, 120, 209, 208, false, (int)Buttons.MarbleHouseWithPatio);
				this.AddRadio(300, 200, 209, 208, false, (int)Buttons.SmallStoneTower);
				this.AddRadio(300, 120, 209, 208, false, (int)Buttons.SandstoneHouseWithPatio);
				this.AddButton(311, 331, 247, 248, (int)Buttons.OkButton, GumpButtonType.Reply, 0);
				this.AddLabel(100, 4, 32, @"Vyber si svuj model domecku");
				this.AddItem(124, 71, 8909);
				this.AddItem(100, 51, 8907);
				this.AddItem(146, 49, 8908);
				this.AddItem(122, 30, 8906);
				this.AddItem(122, 137, 8961);
				this.AddItem(100, 119, 8962);
				this.AddItem(145, 117, 8964);
				this.AddItem(122, 100, 8963);
				this.AddItem(122, 217, 8956);
				this.AddItem(100, 199, 8957);
				this.AddItem(144, 198, 8959);
				this.AddItem(122, 180, 8958);
				this.AddItem(124, 281, 8951);
				this.AddItem(100, 268, 8952);
				this.AddItem(145, 275, 8954);
				this.AddItem(122, 261, 8953);
				this.AddItem(232, 67, 8930);
				this.AddItem(210, 47, 8931);
				this.AddItem(254, 48, 8932);
				this.AddItem(232, 30, 8933);
				this.AddItem(232, 148, 8943);
				this.AddItem(210, 126, 8944);
				this.AddItem(254, 133, 8945);
				this.AddItem(232, 107, 8946);
			}
			
			public enum Buttons
			{
				StoneAndPlaster,
				TwoStoryWoodAndPlaster,
				Brick,
				SmallMarbleWorkshop,
				SmallStoneWorkshop,
				ThatchedRoof,
				WoodAndPlaster,
				Wooden,
				SmallBrick,
				FieldStone,
				LargeHouseWithPatio,
				Tower,
				TwoStoryStoneAndPlaster,
				TwoStoryLogCabin,
				Castle,
				SmallStoneKeep,
				TwoStoryVilla,
				MarbleHouseWithPatio,
				SmallStoneTower,
				SandstoneHouseWithPatio,
				OkButton
			}

			public override void OnResponse( NetState sender, RelayInfo info )
			{
				Mobile from = sender.Mobile;

				switch( info.ButtonID )
				{
					case (int)Buttons.OkButton:
						{
							if (info.IsSwitched( (int)Buttons.StoneAndPlaster) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.StoneAndPlaster;
							}
							else if (info.IsSwitched( (int)Buttons.TwoStoryWoodAndPlaster) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.TwoStoryWoodAndPlaster;
							}
							else if (info.IsSwitched( (int)Buttons.Brick) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.Brick;
							}
							else if (info.IsSwitched( (int)Buttons.SmallMarbleWorkshop) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.SmallMarbleWorkshop;
							}
							else if (info.IsSwitched( (int)Buttons.SmallStoneWorkshop) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.SmallStoneWorkshop;
							}
							else if (info.IsSwitched( (int)Buttons.ThatchedRoof) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.ThatchedRoof;
							}
							else if (info.IsSwitched( (int)Buttons.WoodAndPlaster) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.WoodAndPlaster;
							}
							else if (info.IsSwitched( (int)Buttons.Wooden) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.Wooden;
							}
							else if (info.IsSwitched( (int)Buttons.SmallBrick) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.SmallBrick;
							}
							else if (info.IsSwitched( (int)Buttons.FieldStone) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.FieldStone;
							}
							else if (info.IsSwitched( (int)Buttons.LargeHouseWithPatio) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.LargeHouseWithPatio;
							}
							else if (info.IsSwitched( (int)Buttons.Tower) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.Tower;
							}
							else if (info.IsSwitched( (int)Buttons.TwoStoryStoneAndPlaster) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.TwoStoryStoneAndPlaster;
							}
							else if (info.IsSwitched( (int)Buttons.TwoStoryLogCabin) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.TwoStoryLogCabin;
							}
							else if (info.IsSwitched( (int)Buttons.MarbleHouseWithPatio) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.MarbleHouseWithPatio;
							}
							else if (info.IsSwitched( (int)Buttons.Castle) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.Castle;
							}
							else if (info.IsSwitched( (int)Buttons.TwoStoryStoneAndPlaster) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.TwoStoryStoneAndPlaster;
							}
							else if (info.IsSwitched( (int)Buttons.SmallStoneKeep) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.SmallStoneKeep;
							}
							else if (info.IsSwitched( (int)Buttons.TwoStoryVilla) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.TwoStoryVilla;
							}
							else if (info.IsSwitched( (int)Buttons.SmallStoneTower) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.SmallStoneTower;
							}
							else if (info.IsSwitched( (int)Buttons.SandstoneHouseWithPatio) )
							{
								if( info.Switches.Length == 1 )
									m_MiniHouse.Type = MiniHouseType.SandstoneHouseWithPatio;
							}
						}
						break;
				}
			}
		}
		
		private class TargetInternal : Target
		{
			private MiniHouseDeed m_Deed;

			public TargetInternal( MiniHouseDeed deed ) : base( -1, true, TargetFlags.None )
			{
				m_Deed = deed;

				CheckLOS = false;
			}

			protected override void OnTarget( Mobile from, object targeted )
			{
				IPoint3D p = targeted as IPoint3D;
				Map map = from.Map;

				if ( p == null || map == null || m_Deed.Deleted )
					return;

				if ( m_Deed.IsChildOf( from.Backpack ) )
				{
					BaseAddon addon = m_Deed.Addon;

					Server.Spells.SpellHelper.GetSurfaceTop( ref p );

					ArrayList houses = null;

					AddonFitResult res = addon.CouldFit( p, map, from, ref houses );

					if ( res == AddonFitResult.Valid )
						addon.MoveToWorld( new Point3D( p ), map );
					else if ( res == AddonFitResult.Blocked )
						from.SendLocalizedMessage( 500269 ); // You cannot build that there.
					else if ( res == AddonFitResult.NotInHouse )
						from.SendLocalizedMessage( 500274 ); // You can only place this in a house that you own!
					else if ( res == AddonFitResult.DoorsNotClosed )
						from.SendMessage( "You must close all house doors before placing this." );
					else if ( res == AddonFitResult.DoorTooClose )
						from.SendLocalizedMessage( 500271 ); // You cannot build near the door.
					else if ( res == AddonFitResult.NoWall )
						from.SendLocalizedMessage( 500268 ); // This object needs to be mounted on something.
					
					if ( res == AddonFitResult.Valid )
					{
						m_Deed.Delete();

						if ( houses != null )
						{
							foreach ( Server.Multis.BaseHouse h in houses )
								h.Addons.Add( addon );
						}
					}
					else
					{
						addon.Delete();
					}
				}
				else
				{
					from.SendLocalizedMessage( 1042001 ); // That must be in your pack for you to use it.
				}
			}
		}
		
		public override void GetContextMenuEntries( Mobile from, List<ContextMenuEntry> list )
		{
			base.GetContextMenuEntries( from, list );

			if ( m_Type != MiniHouseType.NoSelected )
				list.Add( new MiniHouseDeedEntry( from, this ) );
		}
		
		public class MiniHouseDeedEntry : ContextMenuEntry
		{
			private Mobile m_From;
			private MiniHouseDeed m_MiniHouseDeed;

			public MiniHouseDeedEntry( Mobile from, MiniHouseDeed deed ) : base( 0154, 5 )
			{
				m_From = from;
				m_MiniHouseDeed = deed;
			}

			public override void OnClick()
			{
				m_MiniHouseDeed.Type = MiniHouseType.NoSelected;
			}
		}

		public MiniHouseDeed( Serial serial ) : base( serial )
		{
		}

		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 0 ); // version

			writer.Write( (int) m_Type );
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 0:
					{
						m_Type = (MiniHouseType)reader.ReadInt();
						break;
					}
			}

			if ( Weight == 0.0 )
				Weight = 1.0;
		}
	}

	public enum MiniHouseType
	{
		NoSelected,
		StoneAndPlaster,
		FieldStone,
		SmallBrick,
		Wooden,
		WoodAndPlaster,
		ThatchedRoof,
		Brick,
		TwoStoryWoodAndPlaster,
		TwoStoryStoneAndPlaster,
		Tower,
		SmallStoneKeep,
		Castle,
		LargeHouseWithPatio,
		MarbleHouseWithPatio,
		SmallStoneTower,
		TwoStoryLogCabin,
		TwoStoryVilla,
		SandstoneHouseWithPatio,
		SmallStoneWorkshop,
		SmallMarbleWorkshop
	}

	public class MiniHouseInfo
	{
		private int[] m_Graphics;
		private int m_LabelNumber;

		public int[] Graphics{ get{ return m_Graphics; } }
		public int LabelNumber{ get{ return m_LabelNumber; } }

		public MiniHouseInfo( int start, int count, int labelNumber )
		{
			m_Graphics = new int[count];

			for ( int i = 0; i < count; ++i )
				m_Graphics[i] = start + i;

			m_LabelNumber = labelNumber;
		}

		public MiniHouseInfo( int labelNumber, params int[] graphics )
		{
			m_LabelNumber = labelNumber;
			m_Graphics = graphics;
		}

		private static MiniHouseInfo[] m_Info = new MiniHouseInfo[]
		{
			/* Stone and plaster house           */ new MiniHouseInfo( 0x22C4, 1, 1011303 ),
			/* Field stone house                 */ new MiniHouseInfo( 0x22DE, 1, 1011304 ),
			/* Small brick house                 */ new MiniHouseInfo( 0x22DF, 1, 1011305 ),
			/* Wooden house                      */ new MiniHouseInfo( 0x22C9, 1, 1011306 ),
			/* Wood and plaster house            */ new MiniHouseInfo( 0x22E0, 1, 1011307 ),
			/* Thatched-roof cottage             */ new MiniHouseInfo( 0x22E1, 1, 1011308 ),
			/* Brick house                       */ new MiniHouseInfo( 1011309, 0x22CD, 0x22CB, 0x22CC, 0x22CA ),
			/* Two-story wood and plaster house  */ new MiniHouseInfo( 1011310, 0x2301, 0x2302, 0x2304, 0x2303 ),
			/* Two-story stone and plaster house */ new MiniHouseInfo( 1011311, 0x22FC, 0x22FD, 0x22FF, 0x22FE ),
			/* Tower                             */ new MiniHouseInfo( 1011312, 0x22F7, 0x22F8, 0x22FA, 0x22F9 ),
			/* Small stone keep                  */ new MiniHouseInfo( 0x22E6, 9, 1011313 ),
			/* Castle                            */ new MiniHouseInfo( 1011314, 0x22CE, 0x22D0, 0x22D2, 0x22D7, 0x22CF, 0x22D1, 0x22D4, 0x22D9, 0x22D3, 0x22D5, 0x22D6, 0x22DB, 0x22D8, 0x22DA, 0x22DC, 0x22DD ),
			/* Large house with patio            */ new MiniHouseInfo( 0x22E2, 4, 1011315 ),
			/* Marble house with patio           */ new MiniHouseInfo( 0x22EF, 4, 1011316 ),
			/* Small stone tower                 */ new MiniHouseInfo( 0x22F5, 1, 1011317 ),
			/* Two-story log cabin               */ new MiniHouseInfo( 0x22FB, 1, 1011318 ),
			/* Two-story villa                   */ new MiniHouseInfo( 0x2300, 1, 1011319 ),
			/* Sandstone house with patio        */ new MiniHouseInfo( 0x22F3, 1, 1011320 ),
			/* Small stone workshop              */ new MiniHouseInfo( 0x22F6, 1, 1011321 ),
			/* Small marble workshop             */ new MiniHouseInfo( 0x22F4, 1, 1011322 )
		};

		public static MiniHouseInfo GetInfo( MiniHouseType type )
		{
			int v = (int)type - 1;

			if ( v < 0 || v >= m_Info.Length )
				v = 0;

			return m_Info[v];
		}
	}
}
