using System;
using System.Collections;
using Server.Mobiles;
using Server.Spells.Necromancy;
using Server.Network;

namespace Server.Items
{
	/// <summary>
	/// Make your opponent bleed profusely with this wicked use of your weapon.
	/// When successful, the target will bleed for several seconds, taking damage as time passes for up to ten seconds.
	/// The rate of damage slows down as time passes, and the blood loss can be completely staunched with the use of bandages. 
	/// </summary>
	public class BleedAttack : WeaponAbility
	{
		public BleedAttack()
		{
		}

		public override int BaseMana { get { return 30; } }

		public override void OnHit(Mobile attacker, Mobile defender, int damage)
		{
			if (!Validate(attacker) || !CheckMana(attacker, true))
				return;

			ClearCurrentAbility(attacker);

			// Necromancers under Lich or Wraith Form are immune to Bleed Attacks.
			TransformContext context = TransformationSpell.GetContext(defender);

			if (((context != null && (context.Type == typeof(LichFormSpell) || context.Type == typeof(WraithFormSpell))) ||
				(defender is BaseCreature && ((BaseCreature)defender).BleedImmune) || DaemonHelm.FullDeamonSetPresent(defender))
				&& !CrusadersShield.FullCrusadersSetPresent(attacker))
			{
				attacker.SendLocalizedMessage(1062052); // Your target is not affected by the bleed attack!
				return;
			}

			attacker.SendLocalizedMessage(1060159); // Your target is bleeding!
			defender.SendLocalizedMessage(1060160); // You are bleeding!

			if (defender is PlayerMobile)
			{
				defender.LocalOverheadMessage(MessageType.Regular, 0x21, 1060757); // You are bleeding profusely
				defender.NonlocalOverheadMessage(MessageType.Regular, 0x21, 1060758, defender.Name); // ~1_NAME~ is bleeding profusely
			}

			defender.PlaySound(0x133);
			defender.FixedParticles(0x377A, 244, 25, 9950, 31, 0, EffectLayer.Waist);

			BeginBleed(defender, attacker);
			CrusadersShield.Bloodbath(attacker);
		}

		private static Hashtable m_Table = new Hashtable();

		public static bool IsBleeding(Mobile m)
		{
			return m_Table.Contains(m);
		}

		public static void BeginBleed(Mobile m, Mobile from)
		{
			Timer t = (Timer)m_Table[m];

			if (t != null)
				t.Stop();

			t = new InternalTimer(from, m);
			m_Table[m] = t;

			t.Start();

			BuffInfo.AddBuff(m, new BuffInfo(BuffIcon.Bleed, 1075829, "5\t25\t2"));
		}

		public static void DoBleed(Mobile m, Mobile from, int level, bool crusader)
		{
			if (m.Alive)
			{
                int damage = 5;

                if (crusader)
                {
                    damage = 100;

                    if (from != null && from.Alive)
                    {
                        if (from.Weapon is CrusadersWarMace)
                        {
                            damage = ((CrusadersWarMace)from.Weapon).ComputeDamage(from, m);
                        }

                        m.Damage(damage, from);
                        from.Hits += AOS.Scale(from.HitsMax, 8);
                    }
                    else
                        m.Damage(damage);
                }
                else
                {
                    damage = Utility.RandomMinMax(level, level * 4);

                    if (!m.Player && m.Map != Map.Underworld)
                        damage *= 2;

                    if (m.Player && m.Map != Map.Trammel && m.Map != Map.Underworld)
                        damage *= 7;

                    AOS.Damage(m, from, damage, true, 100, 0, 0, 0, 0);
                }

                m.PlaySound(0x133);

				Blood blood = new Blood();
				blood.ItemID = Utility.Random(0x122A, 5);
				blood.MoveToWorld(m.Location, m.Map);
			}
			else
			{
				EndBleed(m, false);
			}
		}

		public static void EndBleed(Mobile m, bool message)
		{
			Timer t = (Timer)m_Table[m];

			if (t == null)
				return;

			t.Stop();
			m_Table.Remove(m);

			m.SendLocalizedMessage(1060167); // The bleeding wounds have healed, you are no longer bleeding!
			BuffInfo.RemoveBuff(m, BuffIcon.Bleed);
		}

		private class InternalTimer : Timer
		{
			private Mobile m_From;
			private Mobile m_Mobile;
			private int m_Count;

            public InternalTimer(Mobile from, Mobile m) : base(TimeSpan.FromSeconds(2.0), TimeSpan.FromSeconds(2.0))
			{
				m_From = from;
				m_Mobile = m;
				Priority = TimerPriority.TwoFiftyMS;
			}

			protected override void OnTick()
			{
                DoBleed(m_Mobile, m_From, 5 - m_Count, CrusadersShield.FullCrusadersSetPresent(m_From));

				if (++m_Count == 5)
					EndBleed(m_Mobile, true);
			}
		}
	}
}