using System;
using Server.Items;
using Server.Network;
using Server.Spells;
using Server.Mobiles;
using System.Collections.Generic;
using System.Collections;

namespace Server.Items
{
	public abstract class BaseRanged : BaseMeleeWeapon
	{
		public abstract int EffectID{ get; }
		public abstract Type AmmoType{ get; }
		public abstract Item Ammo{ get; }

		public override int DefHitSound{ get{ return 0x234; } }
		public override int DefMissSound{ get{ return 0x238; } }

		public override SkillName DefSkill{ get{ return SkillName.Archery; } }
		public override WeaponType DefType{ get{ return WeaponType.Ranged; } }
		public override WeaponAnimation DefAnimation{ get{ return WeaponAnimation.ShootXBow; } }

		public override SkillName AccuracySkill{ get{ return SkillName.Archery; } }

		#region Mondain's Legacy
		private bool m_Balanced;
		
		[CommandProperty( AccessLevel.GameMaster )]
		public bool Balanced
		{
			get{ return m_Balanced; }
			set{ m_Balanced = value; InvalidateProperties(); }
		}
		#endregion

		public BaseRanged( int itemID ) : base( itemID )
		{
		}

		public BaseRanged( Serial serial ) : base( serial )
		{
		}

		public override TimeSpan OnSwing( Mobile attacker, Mobile defender )
		{
			WeaponAbility a = WeaponAbility.GetCurrentAbility( attacker );

			// Make sure we've been standing still for .25/.5/1 second depending on Era
			if ( DateTime.Now > (attacker.LastMoveTime + TimeSpan.FromSeconds( 0.5 )) || (Core.AOS && WeaponAbility.GetCurrentAbility( attacker ) is MovingShot) || (m_Balanced) )
			{
				bool canSwing = true;

				if ( Core.AOS )
				{
					canSwing = ( !attacker.Paralyzed && !attacker.Frozen );

					if ( canSwing )
					{
						Spell sp = attacker.Spell as Spell;

						canSwing = ( sp == null || !sp.IsCasting || !sp.BlocksMovement );
					}
				}

				if ( canSwing && attacker.HarmfulCheck( defender ) )
				{
					attacker.DisruptiveAction();
					attacker.Send( new Swing( 0, attacker, defender ) );

					if ( attacker is BaseCreature )
					{
						BaseCreature bc = (BaseCreature)attacker;
						WeaponAbility ab = bc.GetWeaponAbility();
						
						if ( ab != null )
						{
							if ( bc.WeaponAbilityChance > Utility.RandomDouble() )
								WeaponAbility.SetCurrentAbility( bc, ab );
							else
								WeaponAbility.ClearCurrentAbility( bc );
						}
					}

					if ( OnFired( attacker, defender ) )
					{
						if ( CheckHit( attacker, defender ) )
							OnHit( attacker, defender );
						else
							OnMiss( attacker, defender );
					}
				}

				attacker.RevealingAction();

				return GetDelay( attacker );
			}
			else
			{
				attacker.RevealingAction();

				return TimeSpan.FromSeconds( 0.25 );
			}
		}

		public override void OnHit( Mobile attacker, Mobile defender )
		{
          //TALENT
          if ( attacker is PlayerMobile && defender is BaseCreature && attacker.Map != Map.Underworld && Utility.RandomDouble() < 0.2 )
          {
             PlayerMobile.TalentEntry te = ((PlayerMobile)attacker).GetTalent( PlayerMobile.TalentType.SprchaStrel );
             
             if ( te != null && te.Enabled && te.Charges >= 20 )
             {
                ArrayList targets = Utilitky.GetPlayerTargets( defender, 3 );
                int count = Math.Min( targets.Count, 10 );
                
                if ( count > 0 )
                {
            			for (int i = 0; i < count; ++i)
            			{
            				Mobile m = (Mobile)targets[i];
            				
            				if ( m == null || m.Deleted || !m.Alive )
            					continue;
            					
            				if ( CheckHit( attacker, m ) )
                        {
                           PlayHurtAnimation( m );
                           m.PlaySound(GetHitDefendSound(attacker, m));
                           int damage = AOS.Scale( ComputeDamage(attacker, m), 30 );
                           
                           if (m is BaseCreature)
                              ((BaseCreature)m).AlterMeleeDamageFrom(attacker, ref damage);
                          
                           AddBlood(attacker, m, damage);
                           m.Damage( AOS.Damage(m, attacker, damage, false, 20, 20, 20, 20, 20, false, false, 1), attacker );
                        }
            				else
            					OnMiss( attacker, m );
            				
            				attacker.MovingEffect( m, EffectID, 18, 1, false, false );
            			}
                   
                   te.Charges -= 20;
                   
                  if ( te.Charges < 20 )
                     te.Enabled = false;
                }
             }
          }
            
			base.OnHit( attacker, defender );
		}
 
		// 		public override void OnMiss( Mobile attacker, Mobile defender )
		// 		{
		// 			if ( attacker.Player && 0.4 >= Utility.RandomDouble() )
		// 				Ammo.MoveToWorld( new Point3D( defender.X + Utility.RandomMinMax( -1, 1 ), defender.Y + Utility.RandomMinMax( -1, 1 ), defender.Z ), defender.Map );
		// 
		// 			base.OnMiss( attacker, defender );
		// 		}

		public virtual bool OnFired( Mobile attacker, Mobile defender )
		{
			if( !attacker.Player )
			{
				attacker.MovingEffect( defender, EffectID, 18, 1, false, false );
				return true;
			}
         
         Container pack = attacker.Backpack;
			
			if ( pack == null )
			   return false;
			   
			BaseQuiver quiver = attacker.FindItemOnLayer( Layer.MiddleTorso ) as BaseQuiver;
			// 			BaseQuiver quiver = attacker.FindItemOnLayer( Layer.Cloak ) as BaseQuiver;
			
			if( attacker.Player && quiver != null && quiver.LowerAmmoCost > Utility.Random( 99 ) )
			{
				attacker.MovingEffect( defender, EffectID, 18, 1, false, false );
				return true;
			}
			
			if( attacker.Player && quiver != null && quiver.ConsumeAmmo( AmmoType, 1 ) )
			{
				attacker.MovingEffect( defender, EffectID, 18, 1, false, false );
				return true;
			}
			
			BaseQuiver packQuiver = pack.FindItemByType( typeof(BaseQuiver) ) as BaseQuiver;
			
			if( attacker.Player && packQuiver != null && packQuiver.ConsumeAmmo( AmmoType, 1 ) )
			{
				attacker.MovingEffect( defender, EffectID, 18, 1, false, false );
				return true;
			}
			
			if( attacker.Player && pack != null && pack.ConsumeTotal( AmmoType, 1 ) )
			{
				attacker.MovingEffect( defender, EffectID, 18, 1, false, false );
				return true;
			}
			
			return false;
		}
		
		public override void Serialize( GenericWriter writer )
		{
			base.Serialize( writer );

			writer.Write( (int) 3 ); // version
			
			#region Mondain's Legacy ver 3
			writer.Write( (bool) m_Balanced );
			#endregion
		}

		public override void Deserialize( GenericReader reader )
		{
			base.Deserialize( reader );

			int version = reader.ReadInt();

			switch ( version )
			{
				case 3:
					#region Mondain's Legacy
					m_Balanced = reader.ReadBool();
					#endregion
					goto case 2;
				case 2:
				case 1:
					{
						break;
					}
				case 0:
					{
						/*m_EffectID =*/ reader.ReadInt();
						break;
					}
			}

			if ( version < 2 )
			{
				WeaponAttributes.MageWeapon = 0;
				WeaponAttributes.UseBestSkill = 0;
			}
		}
	}
}
