using System;
using System.Collections;
using Server.Items;

namespace Server.Spells.Spellweaving
{
	public class ImmolatingWeaponSpell : ArcanistSpell
	{
		private static SpellInfo m_Info = new SpellInfo(
				"Immolating Weapon", "Thalshara",
				SpellCircle.Second, // 0.5 + 0.5 = 1s base cast delay
				-1
			);

		public override double RequiredSkill { get { return 10.0; } }
		public override int RequiredMana { get { return 32; } }

		public ImmolatingWeaponSpell( Mobile caster, Item scroll ) : base( caster, scroll, m_Info )
		{
		}

		public override bool CheckCast()
		{
			BaseWeapon weapon = Caster.Weapon as BaseWeapon;

			if( weapon == null || weapon is Fists )
			{
				Caster.SendLocalizedMessage( 1060179 ); // You must be wielding a weapon to use this ability!
				return false;
			}

			return base.CheckCast();
		}
		
		public override void OnCast()
		{
			BaseWeapon weapon = Caster.Weapon as BaseWeapon;

			if( weapon == null || weapon is Fists )
			{
				Caster.SendLocalizedMessage( 1060179 ); // You must be wielding a weapon to use this ability!
			}
			else if( CheckSequence() )
			{
				Caster.PlaySound( 0x5CA );
				Caster.FixedParticles( 0x36BD, 20, 10, 5044, EffectLayer.Head );

				/* Tested in Osi.
                 * 
                 * Test values:
                 * 
                 * Without Arcane Focus:
                 *  20.0 =  5 points, 10 secs.
				 * 100.0 =  9 points, 14 secs.
				 * 120.0 = 10 points, 15 secs.
                 * 
                 * With Arcane Focus Level One:
				 *  20.0 =  6 points, 11 secs.
				 * 120.0 = 11 points, 16 secs.
				 */
				int level = GetFocusLevel();
				double skill = Caster.Skills[SkillName.Spellweaving].Value;

				int duration = 10 + (int)( Math.Floor( skill / 24 ) ) + level; // 10 + Floor( Skill / 24 ) + FocusLevel
				int damage = 10 + (int)( Math.Floor( skill / 24 ) ) + level; // 5 + Floor( Skill / 24 ) + FocusLevel

				ImmolatingWeaponEntry iEntry = m_Table[weapon] as ImmolatingWeaponEntry;

				if( iEntry != null )
					iEntry.m_Timer.Stop();

				weapon.Immolating = true;

				m_Table[weapon] = new ImmolatingWeaponEntry( damage, weapon, duration );
			}

			FinishSequence();
		}

		private static Hashtable m_Table = new Hashtable();

		public class ImmolatingWeaponEntry
		{
			public int m_Damage;
			public Timer m_Timer;

			public ImmolatingWeaponEntry( int damage, BaseWeapon weapon, int duration )
			{
				m_Damage = damage;

				m_Timer = new ExpireTimer( weapon, TimeSpan.FromSeconds( duration ) );
				m_Timer.Start();
			}
		}

		public static int GetImmolatingDamage( BaseWeapon w )
		{
			ImmolatingWeaponEntry iEntry = m_Table[w] as ImmolatingWeaponEntry;

			if( iEntry != null )
				return iEntry.m_Damage;

			return 0;
		}

		private class ExpireTimer : Timer
		{
			private BaseWeapon m_Weapon;

			public ExpireTimer( BaseWeapon weapon, TimeSpan delay ) : base( delay )
			{
				m_Weapon = weapon;
				Priority = TimerPriority.OneSecond;
			}

			protected override void OnTick()
			{
				DoExpire();
			}

			public void DoExpire()
			{
				m_Weapon.Immolating = false;
				m_Table.Remove( m_Weapon );
			}
		}
	}
}