using System;
using Server.Targeting;
using Server.Network;
using Server.Misc;
using Server.Items;

namespace Server.Spells.Third
{
	public class WallOfStoneSpell : Spell
	{
		private static SpellInfo m_Info = new SpellInfo(
				"Wall of Stone", "In Sanct Ylem",
				SpellCircle.Third,
				227,
				9011,
				false,
				Reagent.Bloodmoss,
				Reagent.Garlic
			);

		public WallOfStoneSpell( Mobile caster, Item scroll ) : base( caster, scroll, m_Info )
		{
		}

		public override void OnCast()
		{
			Caster.Target = new InternalTarget( this );
		}

		public void Target( IPoint3D p )
		{
			if ( !Caster.CanSee( p ) )
			{
				Caster.SendLocalizedMessage( 500237 ); // Target can not be seen.
			}
			else if ( SpellHelper.CheckTown( p, Caster ) && CheckSequence() )
			{
				SpellHelper.Turn( Caster, p );

				SpellHelper.GetSurfaceTop( ref p );

				int dx = Caster.Location.X - p.X;
				int dy = Caster.Location.Y - p.Y;
				int rx = (dx - dy) * 44;
				int ry = (dx + dy) * 44;

				bool eastToWest;

				if ( rx >= 0 && ry >= 0 )
				{
					eastToWest = false;
				}
				else if ( rx >= 0 )
				{
					eastToWest = true;
				}
				else if ( ry >= 0 )
				{
					eastToWest = true;
				}
				else
				{
					eastToWest = false;
				}
				
				int itemID = 0x80;
				int hue = 0;
				int size = 3;

				Effects.PlaySound( p, Caster.Map, 0x1F6 );
				
            if ( Caster.Player && Caster.Backpack != null && Caster.Skills[SkillName.Lumberjacking].Base >= 100.0 && Caster.Skills[SkillName.Carpentry].Base >= 100.0 )
            {
               BaseCharm charm = (BaseCharm)Caster.Backpack.FindItemByType(typeof(MagicWallToolKit));
               
               if ( charm != null && charm.Charges > 0 && charm.BlessedFor == Caster )
               {
                  charm.Charges--;
                  itemID = eastToWest ? 546 : 545;
                  hue = Utility.RandomNeutralHue();
                  Effects.PlaySound( p, Caster.Map, 573 );
                  size = 5;
               }
            }

				for ( int i = -size; i <= size; ++i )
				{
					Point3D loc = new Point3D( eastToWest ? p.X + i : p.X, eastToWest ? p.Y : p.Y + i, p.Z );
					bool canFit = SpellHelper.AdjustField( ref loc, Caster.Map, 22, true );

					//Effects.SendLocationParticles( EffectItem.Create( loc, Caster.Map, EffectItem.DefaultDuration ), 0x376A, 9, 10, 5025 );

					if ( !canFit )
						continue;

					Item item = new InternalItem( loc, Caster.Map, Caster, itemID, hue );

					Effects.SendLocationParticles( item, 0x376A, 9, 10, 5025 );

					//new InternalItem( loc, Caster.Map, Caster );
				}
			}

			FinishSequence();
		}

		[DispellableField]
		private class InternalItem : Item
		{
			private Timer m_Timer;
			private DateTime m_End;
			private TimeSpan m_Duration = TimeSpan.FromSeconds( 20.0 );

			public override bool BlocksFit{ get{ return true; } }

			public InternalItem( Point3D loc, Map map, Mobile caster, int itemID, int hue ) : base( itemID )
			{
				Visible = false;
				Movable = false;
				Hue = hue;				
				
				if ( itemID != 0x80 )
				{
               m_Duration = TimeSpan.FromSeconds( 40.0 );
            }

				MoveToWorld( loc, map );

				if ( caster.InLOS( this ) )
					Visible = true;
				else
					Delete();

				if ( Deleted )
					return;

				m_Timer = new InternalTimer( this, m_Duration );
				m_Timer.Start();

				m_End = DateTime.Now + m_Duration;
			}

			public InternalItem( Serial serial ) : base( serial )
			{
			}

			public override void Serialize( GenericWriter writer )
			{
				base.Serialize( writer );

				writer.Write( (int) 1 ); // version

				writer.WriteDeltaTime( m_End );
			}

			public override void Deserialize( GenericReader reader )
			{
				base.Deserialize( reader );

				int version = reader.ReadInt();

				switch ( version )
				{
					case 1:
					{
						m_End = reader.ReadDeltaTime();

						m_Timer = new InternalTimer( this, m_End - DateTime.Now );
						m_Timer.Start();

						break;
					}
					case 0:
					{
						m_Timer = new InternalTimer( this, m_Duration );
						m_Timer.Start();

						m_End = DateTime.Now + m_Duration;

						break;
					}
				}
			}

			public override void OnAfterDelete()
			{
				base.OnAfterDelete();

				if ( m_Timer != null )
					m_Timer.Stop();
			}

			private class InternalTimer : Timer
			{
				private InternalItem m_Item;

				public InternalTimer( InternalItem item, TimeSpan duration ) : base( duration )
				{
					Priority = TimerPriority.OneSecond;
					m_Item = item;
				}

				protected override void OnTick()
				{
					m_Item.Delete();
				}
			}
		}

		private class InternalTarget : Target
		{
			private WallOfStoneSpell m_Owner;

			public InternalTarget( WallOfStoneSpell owner ) : base( 12, true, TargetFlags.None )
			{
				m_Owner = owner;
			}

			protected override void OnTarget( Mobile from, object o )
			{
				if ( o is IPoint3D )
					m_Owner.Target( (IPoint3D)o );
			}

			protected override void OnTargetFinish( Mobile from )
			{
				m_Owner.FinishSequence();
			}
		}
	}
}